<?php
declare(strict_types=1);
namespace src\infrastructure\sql;

use Exception;
use src\utils\Key;
use src\domain\user\User;
use src\domain\user\UserRepository;
use src\domain\user\webmaster\WebmasterCustom;
use src\domain\user\provider\ProviderCustom;

class SqlUserRepository implements UserRepository{

    private $conexion;
    private $fields;
	private $table;

    public function __construct() {

        $this->conexion = new Conexion();
        $this->fields = 'name,type,email,ci,status,user_add,date_add,user_update,date_update';
		$this->table = 'user';
    }

    public function getUser(string $field, $value): User{

        $sql = "SELECT id,".$this->fields.",valid FROM ".$this->table." WHERE ".$field." = ?";
        
		$optiones = array($value);
		if($row = $this->conexion->query($sql,$optiones)) {

			return $this->createUser($row);

		} else {

            throw new Exception('El usuario no se encuentra registrado');

        }

	}
	
	public function login(string $email, string $clave): User{

		$sql = "SELECT id,clave,".$this->fields." FROM ".$this->table." WHERE email=? AND status=?"; 
		$optiones = array($email,1);
		if($row = $this->conexion->query($sql,$optiones)) { 
			
			if(Key::VerificarClave($clave,$row['clave'])){
				return $this->createUser($row);
			}else{
				throw new Exception('Email o clave inv&aacute;lidas');
			}

		} else {
			throw new Exception('Si no puedes ingresar puede ser por las siguientes razones:<br> 1. Email o clave inv&aacute;lidas <br>2. Su usuario ha sido desactivado.');
		}

	}

    public function search(array $parameters = [],array $limits = []): array {

        $users = array();
        $specification = new SqlSearchSpecification($parameters,$limits);

		$sql = "SELECT id,".$this->fields.",valid FROM ".$this->table." ".$specification->toSqlClauses();

		if($lista = $this->conexion->listquery($sql,$specification->getParameters())) {

			foreach($lista as $list){
				$users[] = $this->createUser($list);
			}

		}

        return $users;

    }

    public function totalSearch(array $parameters = []): int {

        $specification = new SqlSearchSpecification($parameters,array());
        $sql = "SELECT COUNT(id) as c FROM ".$this->table." ".$specification->toSqlClauses('id',true,false);
		$total = $this->conexion->query($sql,$specification->getParameters());
		return intval($total['c']);

	}
	
    public function providerAll(): array {

        $users = array();
		$sql = "SELECT id,".$this->fields." FROM ".$this->table." WHERE type != ? AND status = ?";

		if($lista = $this->conexion->listquery($sql,array(1,1))) {

			foreach($lista as $list){
				$users[] = $this->createUser($list);
			}

		}

        return $users;

    }

    public function totalproviderAll(): int {

        $sql = "SELECT COUNT(id) as c FROM ".$this->table." WHERE type !=?  AND status = ?";
		$total = $this->conexion->query($sql,array(1,1));
		return intval($total['c']);

    }

    public function add(User $user): User {

		try{

			$password = new Key($user->getPassword());
			$ci = '';

			if(($user->getType() == 2) || ($user->getType() == 3)){
				$ci = $user->getCi();
			}

			$sql = "INSERT INTO ".$this->table." (clave,".$this->fields.") VALUES (?,?,?,?,?,?,?,NOW(),?,NOW())";
	
			$optiones = array(
				$password->clave,
				$user->getName(),
				$user->getType(),
				$user->getEmail(),
				$ci,
				$user->getStatus(),
				$user->getUserAdd(),
				$user->getUserUpdate()
			);
	
			$saved = $this->conexion->execute($sql,$optiones);
	
			if($saved){
	
				$user->setId( intval($this->conexion->lastId()) );
	
			}
	
			return $user;

		}catch(Exception $e){

			throw new Exception($e->getMessage());

		}


    }

    public function update(User $user): bool {

		try{

			$ci = '';

			if(($user->getType() == 2) || ($user->getType() == 3)){
				$ci = $user->getCi();
			}

			$sql = "UPDATE ".$this->table." SET name = ?, email=?, ci =? ,date_update=NOW(), user_update=? WHERE id=?";
	
			$optiones = array(
				$user->getName(),
				$user->getEmail(),
				$ci,
				$user->getUserUpdate(),
				$user->getId()
			);
			return $this->conexion->execute($sql,$optiones);

		}catch(Exception $e){

			throw new Exception($e->getMessage());

		}


    }

    public function delete(User $user): bool {

        $sql = "DELETE FROM ".$this->table." WHERE id=?";
		$optiones = array($user->getId());
		return $this->conexion->execute($sql,$optiones);

    }

    public function changeStatus(User $user): bool {

        $sql = "UPDATE ".$this->table." SET status=? WHERE id=?";
		$optiones = array($user->getStatus(),$user->getId());
		return $this->conexion->execute($sql,$optiones);

	}

	public function isValidOldPassword(int $idUpdate, string $password): bool{

		$sql = "SELECT id,clave FROM ".$this->table." WHERE id=?";
		$optiones = array($idUpdate);
		if($row = $this->conexion->query($sql,$optiones)) { 
			return Key::VerificarClave($password,$row['clave']);

		}

		return false;

	}

	public function updatePassword(User $user):bool{
			
		$password = new Key($user->getPassword());
		$sql = "UPDATE ".$this->table." SET clave=?, active = 1,user_update=?,date_update = NOW() WHERE id=?";
		$optiones = array($password->clave,$user->getUserUpdate(),$user->getId());
		return $this->conexion->execute($sql,$optiones);
	
	}
	
	public function isValueExist(string $field, $value, int $id): bool{

		$sql = "SELECT COUNT(id) as total FROM ".$this->table." WHERE ".$field."=? AND id!=?";
		$optiones = array($value,$id);
		$row = $this->conexion->query($sql,$optiones);
		return intval($row['total']) > 0;

	}

	public function validUser(User $user): bool{

		$sql = "UPDATE ".$this->table." SET valid=?,user_update=?,date_update = NOW() WHERE id=?";
		$optiones = array($user->getValid(),$user->getId(),$user->getId());
		return $this->conexion->execute($sql,$optiones);

	}

    private function createUser(array $row) : User{

		if(($row['type'] != 2) && ($row['type'] != 3)){

			$user = WebmasterCustom::fromRawData($row);

		}else{

			$user = ProviderCustom::fromRawData($row);

		}


        return $user;

    }
}
