<?php
declare(strict_types=1);
namespace src\infrastructure\sql;

use Exception;
use src\domain\document\Document;
use src\domain\document\DocumentCustom;
use src\domain\document\DocumentRepository;

class SqlDocumentRepository implements DocumentRepository{

    private $conexion;
    private $fields;
	private $table;
    private $fieldsSearch;
	private $tableProvider;
	private $tableCategory;

    public function __construct() {

        $this->conexion = new Conexion();
		$this->fields = 'name,file,type_user,category,date,status,user_add,date_add,user_update,date_update';
		$this->fieldsSearch = 'file.id,file.type_user,file.name,file.file,file.category,file.date,file.status,file.user_add,file.date_add,file.user_update,file.date_update,category.id as idCategory ,category.name as nameCategory, GROUP_CONCAT(file_provider.provider SEPARATOR ",") AS provider';
		$this->table = 'file';
		$this->tableProvider = 'file_provider';
		$this->tableCategory = 'category';
    }

    public function getDocument(string $field, $value): Document{

        $sql = "SELECT ".$this->fieldsSearch." FROM ".$this->table." LEFT JOIN ".$this->tableCategory." ON category.id = file.category LEFT JOIN ".$this->tableProvider." ON file_provider.file = file.id WHERE file.".$field." = ?";
        
		$optiones = array($value);
		if($row = $this->conexion->query($sql,$optiones)) {

			if($row['id'] != null){

				return $this->createDocument($row);

			}else{

				throw new Exception('El archivo no se encuentra registrado');

			}


		} else {

            throw new Exception('El archivo no se encuentra registrado');

        }

	}

    public function search(array $parameters = [],array $limits = []): array {

        $documents = array();
        $specification = new SqlSearchJoinSpecification($parameters,$limits);

		$sql = "SELECT ".$this->fieldsSearch." FROM ".$this->table." LEFT JOIN ".$this->tableCategory." ON category.id = file.category LEFT JOIN ".$this->tableProvider." ON file_provider.file = file.id ".$specification->toSqlClauses('file.id',false);
 
		if($lista = $this->conexion->listquery($sql,$specification->getParameters())) {

			foreach($lista as $list){ 

				if($list['id'] != null){

					$documents[] = $this->createDocument($list);

				}

			}

		}

        return $documents;

	}

    public function totalSearch(array $parameters = []): int {

        $specification = new SqlSearchJoinSpecification($parameters,array());
        $sql = "SELECT COUNT(file.id) as c FROM ".$this->table." LEFT JOIN ".$this->tableCategory." ON category.id = file.category LEFT JOIN ".$this->tableProvider." ON file_provider.file = file.id ".$specification->toSqlClauses('file.id',false,false);
		$total = $this->conexion->query($sql,$specification->getParameters());
		return intval($total['c']);

    }

    public function add(Document $file): Document {

		try{

			$sql = "INSERT INTO ".$this->table." (".$this->fields.") VALUES (?,?,?,?,?,?,?,NOW(),?,NOW())";
	
			$optiones = array(
				$file->getName(),
				$file->getFile(),
				$file->getTypeUser(),
				$file->getCategory()->getId(),
				date('Y-m-d',strtotime($file->getDate())),
				$file->getStatus(),
				$file->getUserAdd(),
				$file->getUserUpdate()
			);
	
			$saved = $this->conexion->execute($sql,$optiones);
	
			if($saved){
	
				$file->setId( $this->conexion->lastId() );
				$this->saveProvider($file);
				
			}
			
			return $file;
			
		}catch(Exception $e){
			
			throw new Exception($e->getMessage());
			
		}
		
		
    }
	
    public function update(Document $document): document {
		
		try{
			
			$sql = "UPDATE ".$this->table." SET name = ?,type_user=?,date=?, category = ?, file = ? ,date_update=NOW(), user_update=? WHERE id=?";
			
			$optiones = array(
				$document->getName(),
				$document->getTypeUser(),
				date('Y-m-d',strtotime($document->getDate())),
				$document->getCategory()->getId(),
				$document->getFile(),
				$document->getUserUpdate(),
				$document->getId()
			);

			
			if($this->conexion->execute($sql,$optiones)){
				
				$this->deleteProviders($document);
				$this->saveProvider($document);

			}

			return $document;
			
		}catch(Exception $e){
			
			throw new Exception($e->getMessage());
			
		}
		
		
	}
	
	private function saveProvider(Document $document){
		
		if( count($document->getProvider()) ){
	
	
			foreach($document->getProvider() as $index => $provider){
	
				$sql = "INSERT INTO ".$this->tableProvider." ( file, provider ) VALUES (?,?)";
	
				$optiones = array(
					$document->getId(),
					$provider
				);
		
				if(!$this->conexion->execute($sql,$optiones)){
	
					unset($document->getProvider()[$index]);
					break;
	
				}
	
			}
	
	
		}
		
	}
	
	private function deleteProviders(Document $document): bool {
		
		$sql = "DELETE FROM ".$this->tableProvider." WHERE file=?";
		$optiones = array($document->getId());
		return $this->conexion->execute($sql,$optiones);
		
	}

    public function delete(Document $document): bool {

        $sql = "DELETE FROM ".$this->table." WHERE id=?";
		$optiones = array($document->getId());
		return $this->conexion->execute($sql,$optiones);

    }

    public function changeStatus(Document $document): bool {

        $sql = "UPDATE ".$this->table." SET status=? WHERE id=?";
		$optiones = array($document->getStatus(),$document->getId());
		return $this->conexion->execute($sql,$optiones);

	}

	public function isValueExist(string $field, $value, int $id): bool{

		$sql = "SELECT COUNT(id) as total FROM ".$this->table." WHERE ".$field."=? AND id!=?";
		$optiones = array($value,$id);
		$row = $this->conexion->query($sql,$optiones);
		return intval($row['total']) > 0;

	}

    private function createDocument(array $row) : Document{

		return DocumentCustom::fromRawData($row);

    }
}
