<?php
declare(strict_types=1);
namespace src\domain\document;

use Exception;
use src\domain\event\LogEventPublisher;
use src\domain\validation\ValidationHandler;

class Document{

    protected $id;
    protected $name;
    protected $category;
    protected $provider;
    protected $typeUser;
    protected $date;
    protected $file;
    protected $status;
    protected $userAdd;
    protected $dateAdd;
    protected $userUpdate;
    protected $dateUpdate;

	public function __construct(
        ?int $id,
        string $name,
        string $file,
        int $categoryId,
        string $categoryName,
        array $provider,
        int $typeUser,
        string $date,
        int $status,
        ?int $userAdd,
        ?int $userUpdate,
        ?string $dateAdd, 
        ?string $dateUpdate
    ){

        $category = new DocumentCategory($categoryId,$categoryName);
        $errors = $this->canSave($name,$file,$category);
        if(!$errors->hasError()){

            $this->id = $id;
            $this->name = $name;
            $this->category = $category;
            $this->provider = $provider;
            $this->typeUser = $typeUser;
            $this->date = $date;
            $this->file = $file;
            $this->status = $status;
            $this->userAdd = $userAdd;
            $this->dateAdd = $dateAdd;
            $this->userUpdate = $userUpdate; 
            $this->dateUpdate = $dateUpdate;
            LogEventPublisher::instance()->publish(
                new DocumentCreated($this->userAdd,$this->name)
            );

        }else{

            throw new Exception( $errors->printErrors() );

        }
        
    }

    public function update( DocumentDTO $documentData){

        $category = new DocumentCategory($documentData->idCategory,$documentData->nameCategory);
        $errors = $this->canSave($documentData->name,$documentData->file,$category);
        if(!$errors->hasError()){

            $this->name = $documentData->name;
            $this->file = $documentData->file;
            $this->category = $category;
            $this->provider = $documentData->provider;
            $this->typeUser = $documentData->typeUser;
            $this->date = $documentData->date;
            $this->userUpdate = $documentData->userUpdate;
            $this->dateUpdate = date('Y-m-d H:i:s');
            LogEventPublisher::instance()->publish(
                new DocumentUpdated($this->userUpdate,$this->name)
            );

        }else{

            throw new Exception( $errors->printErrors() );

        }

    }

    public function changeStatus( int $newStatus , int $idUserUpdate ){

        $this->status = $newStatus;
        $this->userUpdate = $idUserUpdate;
        $this->dateUpdate = date('Y-m-d H:i:s');
        LogEventPublisher::instance()->publish(
            new DocumentUpdated($this->userUpdate,$this->name)
        );

    }
    
    public function delete( int $idUserUpdate ){

        LogEventPublisher::instance()->publish(
            new DocumentDeleted($this->userUpdate,$this->name)
        );

    }

    public static function canSave(string $name) : ValidationHandler{
        
        $categoryValidator = new CategoryValidation();
        return $categoryValidator->canSave($name);
        
    }

    public function getId(){
        return $this->id;
    }

    public function getName(){
        return $this->name;
    }

    public function getDateAdd(){
        return date('d/m/Y',strtotime($this->dateAdd));
    }

    public function getDateUpdate(){
        return date('d/m/Y',strtotime($this->dateUpdate));
    }

    public function setId($id){
        $this->id = $id;
    }

    public function getUserUpdate(){
        return $this->userUpdate;
    }

    public function getUserAdd(){
        return $this->userAdd;
    }

    public function getStatus(){
        return $this->status;
    }

    public function getCategory(){
        return $this->category;
    }
    
    public function getFile(){
        return $this->file;
    }

    public function getProvider(){
        return $this->provider;
    }

    public function getDate(){
        return date('d-m-Y',strtotime($this->date));
    }

    public function getTypeUser(){
        return $this->typeUser;
    }
}
?>
