<?php
declare(strict_types=1);
namespace src\application\user;

use src\domain\user\User;
use src\domain\user\UserDTO;
use src\application\Authorization;
use src\application\Authentication;
use src\domain\user\UserRepository;
use src\domain\event\LogEventPublisher;
use src\domain\user\webmaster\Webmaster;
use src\application\exception\UserInvalid;
use src\application\exception\SessionExpire;
use src\domain\event\LogEventSaveSubscriber;
use src\domain\validation\ValidationHandler;
use src\infrastructure\sql\SqlLogEventStore;

class WebmasterCreate {

    private $userRepository;
    private $authorization;
    private $userLogin;

	public function __construct( UserRepository $userRepository ){

        $this->userRepository = $userRepository;
        $this->authorization = new Authorization();
        $this->userLogin = Authentication::getUserAuth();
        LogEventPublisher::instance()->subscribe(new LogEventSaveSubscriber( new SqlLogEventStore() ));    
    }
    
    
    private function checkAuthorization(){
        
        if( !$this->authorization->isLogin() ){
            
            throw new SessionExpire();
            
        }else{
            
            if(!$this->authorization->isWebmaster()){
                
                throw new UserInvalid();
                
            }
            
        }
        
    }
    
    /**
     * Crea un nuevo usuario
     *
     * @param UserRequest $request
     * @return array
     */
    public function execute( UserRequest $request ) : array{

        $this->checkAuthorization();
        
        try{

            if($this->userRepository->isValueExist( 'email',$request->getEmail(),-1 )){

                throw new \Exception('El email ya se encuentra registrado');

            }
            
            $errors = Webmaster::canSave( $request->getName(),$request->getEmail(),$request->getPassword() );

            if( $errors->hasError() ){

                throw new \Exception($errors->printErrors());
                
            }

            $user = new Webmaster( 
                NULL, 
                $request->getName(), 
                $request->getEmail(),
                $request->getPassword(),
                1,
                $request->getType(),
                $this->userLogin->id,
                $this->userLogin->id,
                date('Y-m-d H:i:s'),
                date('Y-m-d H:i:s')
            );
            $this->userRepository->add( $user );
            return array("success"=>true,"message"=>"Usuario creado exitosamente",'user'=>new UserDTO($user));
            

        }catch(\Exception $e){

			throw new \Exception($e->getMessage());

		}



    }
}
?>
