<?php
declare(strict_types=1);
namespace src\application\user;

use Exception;
use src\infrastructure\sql\SqlLogEventStore;
use src\domain\event\LogEventPublisher;
use src\domain\event\LogEventSaveSubscriber;
use src\domain\user\UserRepository;
use src\domain\validation\ValidationHandler;
use src\domain\user\User;
use src\domain\user\UserDTO;
use src\application\Authorization;
use src\application\Authentication;
use src\application\exception\SessionExpire;
use src\application\exception\UserInvalid;

class UserUpdate {

    private $userRepository;
    private $authorization;
    private $userLogin;

	public function __construct( UserRepository $userRepository ){

        $this->userRepository = $userRepository;
        $this->authorization = new Authorization();
        $this->userLogin = Authentication::getUserAuth();
        LogEventPublisher::instance()->subscribe(new LogEventSaveSubscriber( new SqlLogEventStore() ));    
    }
    
    private function checkAuthorization($idUpdate){

        if( !$this->authorization->isLogin() ){
    
            throw new SessionExpire();
        
        }else{

            if(!$this->authorization->isWebmaster()){

                if($userLogin->id != $idUpdate){

                    throw new UserInvalid();

                }


            }


        }

    }

    private function transformRequestToDTO ( User $user, UserRequest $request ) : UserDTO{

        $userDTO = new UserDTO($user);
        
        $userDTO->name = $request->getName();
        $userDTO->email = $request->getEmail();
        $userDTO->userUpdate = $this->userLogin->id;

        return $userDTO;

    }
    
    /**
     * Modificar los datos de un usuario
     *
     * @param UserRequest $request
     * @return array
     */
    public function execute( UserRequest $request ) : array{

        $this->checkAuthorization($request->getId());

        try{

            if($this->userRepository->isValueExist( 'email',$request->getEmail(),$request->getId() )){

                throw new Exception('El email ya se encuentra registrado');

            }

            $user = $this->userRepository->getUser('id',$request->getId());
            
            $errors = $user->canSave( $request->getName(),$request->getEmail() );
            if( $errors->hasError() ){

                throw new Exception($errors->printErrors());
                
            }

            $user->update( $this->transformRequestToDTO( $user, $request) );

            $this->userRepository->update($user);

            if($request->getPassword() != ''){

                $errors = $user->canChangePassword( $request->getPassword() );
                if( $errors->hasError() ){ 

                    throw new Exception($errors->printErrors());
                    
                }

                $user->changePassword($request->getPassword(),$this->userLogin->id);
                $this->userRepository->updatePassword($user);    

            }

            return array("success"=>true,"message"=>"Usuario modificado exitosamente",'user'=>new UserDTO($user));
            

        }catch(Exception $e){

			throw new Exception($e->getMessage());

		}



    }
}
?>
